<?php
session_start();
header('Content-Type: application/json');

require_once __DIR__ . '/vendor/autoload.php';

use App\Core\Database;

$db = Database::connect();

try {

    $input = json_decode(file_get_contents("php://input"), true);

    $ref = $input['booking_reference'] ?? '';

    if (!$ref) {
        throw new Exception('Invalid booking reference.');
    }

    $db = Database::connect();
    $db->beginTransaction();

    $stmt = $db->prepare("
        SELECT id, status, pickup_datetime
        FROM bookings
        WHERE booking_reference = ?
        LIMIT 1
    ");
    $stmt->execute([$ref]);
    $booking = $stmt->fetch();

    if (!$booking) {
        throw new Exception('Booking not found.');
    }

    if ($booking['status'] !== 'pending') {
        throw new Exception('Booking cannot be cancelled.');
    }

    // Optional: Restrict cancellation if within 6 hours
    $pickupTime = strtotime($booking['pickup_datetime']);
    $now = time();

    $hoursDifference = ($pickupTime - $now) / 3600;

    if ($hoursDifference < 6) {
        throw new Exception('Bookings cannot be cancelled within 6 hours of pickup.');
    }

    // Update booking status
    $stmt = $db->prepare("
        UPDATE bookings
        SET status = 'cancelled'
        WHERE id = ?
    ");
    $stmt->execute([$booking['id']]);

    // Insert timeline entry
    $stmt = $db->prepare("
        INSERT INTO booking_timeline (booking_id, status, updated_by)
        VALUES (?, 'cancelled', 'customer')
    ");
    $stmt->execute([$booking['id']]);

    $db->commit();

    echo json_encode([
        'status' => 'success'
    ]);

} catch (Exception $e) {

    if (isset($db) && $db->inTransaction()) {
        $db->rollBack();
    }

    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}
