<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/auth.php';
require_role(['admin']);

require_once __DIR__ . '/includes/config.php';

/*
  Lightweight QR generator (no composer dependency)
  Using Google Chart API for QR rendering
*/

function generateRandomKey(int $length = 32): string {
    return bin2hex(random_bytes($length / 2));
}

$generatedKey = null;
$keyType = null;
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    csrf_validate($_POST['_csrf'] ?? null);

    $keyType = $_POST['key_type'] ?? '';

    if (!in_array($keyType, ['admin', 'guard_qr'], true)) {
        $error = "Invalid key type.";
    } else {

        $generatedKey = strtoupper(generateRandomKey(32));

        $stmt = db()->prepare("
            INSERT INTO registration_keys (key_type, key_value, status)
            VALUES (?, ?, 'active')
        ");
        $stmt->execute([$keyType, $generatedKey]);
    }
}

include __DIR__ . '/includes/header.php';
?>

<div class="card">
  <div class="card-header">
    <h3 class="card-title">
      <i class="fas fa-qrcode"></i> Generate One-Time QR Key
    </h3>
  </div>
  <div class="card-body">

    <?php if ($error): ?>
      <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <form method="post">
      <input type="hidden" name="_csrf" value="<?= htmlspecialchars(csrf_token()) ?>">

      <div class="form-group">
        <label>Key Type</label>
        <select name="key_type" class="form-control" required>
          <option value="">-- Select --</option>
          <option value="guard_qr">Guard Registration QR</option>
          <option value="admin">Admin Registration QR</option>
        </select>
      </div>

      <button class="btn btn-primary">
        <i class="fas fa-key"></i> Generate QR Key
      </button>
    </form>

    <?php if ($generatedKey): ?>

      <hr>

      <div class="alert alert-success">
        <strong>Generated Key:</strong><br>
        <code style="font-size:16px;"><?= htmlspecialchars($generatedKey) ?></code>
      </div>

      <?php
        $qrData = urlencode($generatedKey);
        $qrUrl = "https://api.qrserver.com/v1/create-qr-code/?size=300x300&data={$qrData}";
      ?>

      <div class="text-center">
        <img src="<?= $qrUrl ?>" alt="QR Code" class="img-fluid mb-3">
        <br>
        <a href="<?= $qrUrl ?>" download="securehrms_qr.png" class="btn btn-success">
          <i class="fas fa-download"></i> Download QR
        </a>
      </div>

      <div class="mt-4">
        <h5>Usage Instructions</h5>
        <ul>
          <li>This QR key is one-time use.</li>
          <li>Status will change to <strong>used</strong> once registered.</li>
          <li>Revoke unused keys manually if required.</li>
        </ul>
      </div>

    <?php endif; ?>

  </div>
</div>

<div class="card mt-4">
  <div class="card-header">
    <h3 class="card-title">Active Keys</h3>
  </div>
  <div class="card-body table-responsive">
    <table class="table table-bordered table-striped">
      <thead>
        <tr>
          <th>ID</th>
          <th>Type</th>
          <th>Key</th>
          <th>Status</th>
          <th>Created</th>
        </tr>
      </thead>
      <tbody>
        <?php
        $keys = db()->query("
          SELECT id, key_type, key_value, status, created_at
          FROM registration_keys
          ORDER BY id DESC
          LIMIT 50
        ")->fetchAll();

        foreach ($keys as $k):
        ?>
          <tr>
            <td><?= (int)$k['id'] ?></td>
            <td><?= htmlspecialchars($k['key_type']) ?></td>
            <td><code><?= htmlspecialchars($k['key_value']) ?></code></td>
            <td>
              <?php if ($k['status'] === 'active'): ?>
                <span class="badge badge-success">Active</span>
              <?php elseif ($k['status'] === 'used'): ?>
                <span class="badge badge-secondary">Used</span>
              <?php else: ?>
                <span class="badge badge-danger">Revoked</span>
              <?php endif; ?>
            </td>
            <td><?= htmlspecialchars($k['created_at']) ?></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>